/*=============================================================================
	UnClass.h: UClass definition.
	Copyright 1997-1999 Epic Games, Inc. All Rights Reserved.

	Revision history:
		* Created by Tim Sweeney.
=============================================================================*/
#ifdef _MSC_VER
#pragma warning( disable : 4121 )
#endif
// vogel: alignment of a member was sensitive to packing

#include "UnForcePacking_begin.h"

/*-----------------------------------------------------------------------------
	Constants.
-----------------------------------------------------------------------------*/

// Boundary to align class properties on.
enum {PROPERTY_ALIGNMENT=sizeof(void*)};

/*-----------------------------------------------------------------------------
	FRepRecord.
-----------------------------------------------------------------------------*/

//
// Information about a property to replicate.
//
struct FRepRecord
{
	UProperty* Property;
	INT Index;
	FRepRecord(UProperty* InProperty,INT InIndex)
	: Property(InProperty), Index(InIndex)
	{}
};

/*-----------------------------------------------------------------------------
	FDependency.
-----------------------------------------------------------------------------*/

//
// One dependency record, for incremental compilation.
//
class CORE_API FDependency
{
public:
	// Variables.
	UClass*		Class;
	UBOOL		Deep;
	DWORD		ScriptTextCRC;

	// Functions.
	FDependency();
	FDependency( UClass* InClass, UBOOL InDeep );
	UBOOL IsUpToDate();
	CORE_API friend FArchive& operator<<( FArchive& Ar, FDependency& Dep );
};

/*-----------------------------------------------------------------------------
	FRepLink.
-----------------------------------------------------------------------------*/

//
// A tagged linked list of replicatable variables.
//
class FRepLink
{
public:
	UProperty*	Property;		// Replicated property.
	FRepLink*	Next;			// Next replicated link per class.
	FRepLink( UProperty* InProperty, FRepLink* InNext )
	:	Property	(InProperty)
	,	Next		(InNext)
	{}
};

/*-----------------------------------------------------------------------------
	FLabelEntry.
-----------------------------------------------------------------------------*/

//
// Entry in a state's label table.
//
struct CORE_API FLabelEntry
{
	// Variables.
	FName	Name;
	INT		iCode;

	// Functions.
	FLabelEntry( FName InName, INT iInCode );
	CORE_API friend FArchive& operator<<( FArchive& Ar, FLabelEntry &Label );
};

/*-----------------------------------------------------------------------------
	UField.
-----------------------------------------------------------------------------*/

//
// Base class of UnrealScript language objects.
//
class CORE_API UField : public UObject
{
	DECLARE_ABSTRACT_CLASS(UField,UObject,0,Core)
	NO_DEFAULT_CONSTRUCTOR(UField)

	// Constants.
	enum {HASH_COUNT = 256};

	// Variables.
	UField*			SuperField;
	UField*			Next;
	UField*			HashNext;

	// Constructors.
	UField( ENativeConstructor, UClass* InClass, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags, UField* InSuperField );
	UField( EStaticConstructor, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags );
	UField( UField* InSuperField );

	// UObject interface.
	void Serialize( FArchive& Ar );
	void PostLoad();
	void Register();

	// UField interface.
	virtual void AddCppProperty( UProperty* Property );
	virtual UBOOL MergeBools();
	virtual void Bind();
	virtual UClass* GetOwnerClass();
	virtual INT GetPropertiesSize() const;
};

/*-----------------------------------------------------------------------------
	UStruct.
-----------------------------------------------------------------------------*/

enum EStructFlags
{
	// State flags.
	STRUCT_Native		= 0x00000001,	
	STRUCT_Export		= 0x00000002,
	// gam ---
	STRUCT_Long	        = 0x00000004, // will get shown as "..." in editactor until expanded.
	STRUCT_Inherit      = STRUCT_Long,
	// --- gam
	STRUCT_Init			= 0x00000008,
};

//
// An UnrealScript structure definition.
//
class CORE_API UStruct : public UField
{
	DECLARE_CLASS(UStruct,UField,0,Core)
	NO_DEFAULT_CONSTRUCTOR(UStruct)

	// Variables.
	UTextBuffer*        ScriptText;
	UTextBuffer*        CppText;
	UField*             Children;
	INT                 PropertiesSize;
	INT                 PropertiesAlignment;
	FName               FriendlyName;
	TArray<BYTE>        Script;
	TArray<BYTE>        ScriptExt;
	UBOOL               X64Support;

	// Compiler info.
	INT                 TextPos;
	INT                 Line;
	DWORD               StructFlags;

	// In memory only.
	UProperty*          RefLink;
	UProperty*          PropertyLink;
	UProperty*          ConfigLink;
	UProperty*          ConstructorLink;

	// Constructors.
	UStruct( ENativeConstructor, INT InSize, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags, UStruct* InSuperStruct );
	UStruct( EStaticConstructor, INT InSize, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags );
	UStruct( UStruct* InSuperStruct );

	// UObject interface.
	void Serialize( FArchive& Ar );
	void PostLoad();
	void Destroy();
	void Register();

	// UField interface.
	void AddCppProperty( UProperty* Property );

	// UStruct interface.
	virtual UStruct* GetInheritanceSuper() {return GetSuperStruct();}
	virtual void Link( FArchive& Ar, UBOOL Props );
	virtual void SerializeBin( FArchive& Ar, BYTE* Data, INT MaxReadBytes );
	virtual void SerializeTaggedProperties( FArchive& Ar, BYTE* Data, UClass* DefaultsClass );
	virtual void CleanupDestroyed( BYTE* Data );
	virtual EExprToken SerializeExpr( INT& iCode, FArchive& Ar );
    void* LoadScriptPtr(INT iCode) const {
        checkSlow(iCode < Script.Num());

        if constexpr (sizeof(void*) != sizeof(DWORD)) {
            if (!X64Support && ScriptExt(iCode) == 0xCB) {
                DWORD Hi, Lo;
                appMemcpy(&Lo, &Script(iCode), sizeof(DWORD));
                appMemcpy(&Hi, &ScriptExt(iCode + 1), sizeof(DWORD));
                return (void*)(QWORD(Lo) | (QWORD(Hi) << (sizeof(DWORD) * CHAR_BIT)));
            }
        }

        void* RetVal = nullptr;
        appMemcpy(&RetVal, &Script(iCode), sizeof(void*));
        return RetVal;
    }
    void  StoreScriptPtr(INT iCode, const void* Ptr) {
        if constexpr (sizeof(void*) != sizeof(DWORD)) {
            if (!X64Support) {
                DWORD Hi = DWORD(PTRINT(Ptr) >> (sizeof(DWORD) * CHAR_BIT));
                DWORD Lo = DWORD(PTRINT(Ptr));
                appMemcpy(&Script(iCode), &Lo, sizeof(Lo));
                ScriptExt(iCode) = 0xCB;
                appMemcpy(&ScriptExt(iCode + 1), &Hi, sizeof(Hi));
                return;
            }
        }

        appMemcpy(&Script(iCode), &Ptr, sizeof(Ptr));
    }
	virtual INT GetPropertiesAlignment() const
	{
		return PropertiesAlignment;
	}
	INT GetPropertiesSize() const
	{
		return PropertiesSize;
	}
	DWORD GetScriptTextCRC() const
	{
		return ScriptText ? appStrCrc(*ScriptText->Text) : 0;
	}
	void SetPropertiesSize( INT NewSize )
	{
		PropertiesSize = NewSize;
	}
	UBOOL IsChildOf( const UStruct* SomeBase ) const
	{
		guardSlow(UStruct::IsChildOf);
		for( const UStruct* Struct=this; Struct; Struct=Struct->GetSuperStruct() )
			if( Struct==SomeBase ) 
				return 1;
		return 0;
		unguardobjSlow;
	}
	virtual FString GetNameCPP() const
	{
		return FString::Printf( TEXT("F%s"), GetName() );
	}
	UStruct* GetSuperStruct() const
	{
		guardSlow(UStruct::GetSuperStruct);
		checkSlow(!SuperField||SuperField->IsA(UStruct::StaticClass()));
		return (UStruct*)SuperField;
		unguardSlow;
	}
	UBOOL StructCompare( const void* A, const void* B );
	FString FunctionMD5();		// Returns the Quick MD5 hash for this package	

protected:

	// Cheat Protection
		
	BYTE FunctionMD5Digest[16];		// Holds a MD5 digest for this function
};

/*-----------------------------------------------------------------------------
	UFunction.
-----------------------------------------------------------------------------*/

//
// An UnrealScript function.
//
class CORE_API UFunction : public UStruct
{
	DECLARE_CLASS(UFunction,UStruct,CLASS_IsAUFunction,Core)
	DECLARE_WITHIN(UState)
	NO_DEFAULT_CONSTRUCTOR(UFunction)

	// Persistent variables.
	DWORD FunctionFlags;
	_WORD iNative;
	_WORD RepOffset;
	BYTE  OperPrecedence;

	// Variables in memory only.
	BYTE  NumParms;
	_WORD ParmsSize;
	_WORD ReturnValueOffset;
	void (UObject::*Func)( FFrame& TheStack, RESULT_DECL );

	// Constructors.
	UFunction( UFunction* InSuperFunction );

	// UObject interface.
	void Serialize( FArchive& Ar );
	void PostLoad();

	// UField interface.
	void Bind();

	// UStruct interface.
	UBOOL MergeBools() {return 0;}
	UStruct* GetInheritanceSuper() {return NULL;}
	void Link( FArchive& Ar, UBOOL Props );

	// UFunction interface.
	UFunction* GetSuperFunction() const
	{
		guardSlow(UFunction::GetSuperFunction);
		checkSlow(!SuperField||SuperField->IsA(UFunction::StaticClass()));
		return (UFunction*)SuperField;
		unguardSlow;
	}
	UProperty* GetReturnProperty();
};

/*-----------------------------------------------------------------------------
	UState.
-----------------------------------------------------------------------------*/

//
// An UnrealScript state.
//
class CORE_API UState : public UStruct
{
	DECLARE_CLASS(UState,UStruct,CLASS_IsAUState,Core)
	NO_DEFAULT_CONSTRUCTOR(UState)

	// Variables.
	QWORD ProbeMask;
	QWORD IgnoreMask;
	DWORD StateFlags;
	_WORD LabelTableOffset;
	UField* VfHash[HASH_COUNT];

	// Constructors.
	UState( ENativeConstructor, INT InSize, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags, UState* InSuperState );
	UState( EStaticConstructor, INT InSize, const TCHAR* InName, const TCHAR* InPackageName, DWORD InFlags );
	UState( UState* InSuperState );

	// UObject interface.
	void Serialize( FArchive& Ar );
	void Destroy();

	// UStruct interface.
	UBOOL MergeBools() {return 1;}
	UStruct* GetInheritanceSuper() {return GetSuperState();}
	void Link( FArchive& Ar, UBOOL Props );

	// UState interface.
	UState* GetSuperState() const
	{
		guardSlow(UState::GetSuperState);
		checkSlow(!SuperField||SuperField->IsA(UState::StaticClass()));
		return (UState*)SuperField;
		unguardSlow;
	}
};

/*-----------------------------------------------------------------------------
	UEnum.
-----------------------------------------------------------------------------*/

//
// An enumeration, a list of names usable by UnrealScript.
//
class CORE_API UEnum : public UField
{
	DECLARE_CLASS(UEnum,UField,0,Core)
	DECLARE_WITHIN(UStruct)
	NO_DEFAULT_CONSTRUCTOR(UEnum)

	// Variables.
	TArray<FName> Names;

	// Constructors.
	UEnum( UEnum* InSuperEnum );

	// UObject interface.
	void Serialize( FArchive& Ar );

	// UEnum interface.
	UEnum* GetSuperEnum() const
	{
		guardSlow(UEnum::GetSuperEnum);
		checkSlow(!SuperField||SuperField->IsA(UEnum::StaticClass()));
		return (UEnum*)SuperField;
		unguardSlow;
	}
};

/*-----------------------------------------------------------------------------
	UClass.
-----------------------------------------------------------------------------*/
#define TRACK_ISA 0
//
// An object class.
//
class CORE_API UClass : public UState
{
	DECLARE_CLASS(UClass,UState,0,Core)
	DECLARE_WITHIN(UPackage)

	// Variables.
	DWORD				ClassFlags;
	INT					ClassUnique;
	FGuid				ClassGuid;
	UClass*				ClassWithin;
	FName				ClassConfigName;
	TArray<FRepRecord>	ClassReps;
	TArray<UField*>		NetFields;
	TArray<FDependency> Dependencies;
	TArray<FName>		PackageImports;
	TArray<BYTE>		Defaults;
	TArray<FName>		HideCategories;
    TArray<FName>       DependentOn; //amb,gam
	void(*ClassConstructor)(void*);
	void(UObject::*ClassStaticConstructor)();

	// In memory only.
	FString				DefaultPropText;
#if TRACK_ISA
    INT IsACount;
#endif

	// Constructors.
	UClass();
	UClass( UClass* InSuperClass );
	UClass( ENativeConstructor, DWORD InSize, DWORD InClassFlags, UClass* InBaseClass, UClass* InWithinClass, FGuid InGuid, const TCHAR* InNameStr, const TCHAR* InPackageName, const TCHAR* InClassConfigName, DWORD InFlags, void(*InClassConstructor)(void*), void(UObject::*InClassStaticConstructor)() );
	UClass( EStaticConstructor, DWORD InSize, DWORD InClassFlags, FGuid InGuid, const TCHAR* InNameStr, const TCHAR* InPackageName, const TCHAR* InClassConfigName, DWORD InFlags, void(*InClassConstructor)(void*), void(UObject::*InClassStaticConstructor)() );

	// UObject interface.
	void Serialize( FArchive& Ar );
	void PostLoad();
	void Destroy();
	void Register();

	// UField interface.
	void Bind();

	// UStruct interface.
	UBOOL MergeBools() {return 1;}
	UStruct* GetInheritanceSuper() {return GetSuperClass();}
	FString GetNameCPP() const
	{
		const UClass* C;
		for( C=this; C; C=C->GetSuperClass() )
			if( appStricmp(C->GetName(),TEXT("Actor"))==0 )
				break;
		return FString::Printf( TEXT("%s%s"), C ? TEXT("A") : TEXT("U"), GetName() );
	}
	void Link( FArchive& Ar, UBOOL Props );

	// UClass interface.
	void AddDependency( UClass* InClass, UBOOL InDeep )
	{
		guard(UClass::AddDependency);
		INT i;
		for( i=0; i<Dependencies.Num(); i++ )
			if( Dependencies(i).Class==InClass )
				Dependencies(i).Deep |= InDeep;
		if( i==Dependencies.Num() )
			new(Dependencies)FDependency( InClass, InDeep );
		unguard;
	}
	UClass* GetSuperClass() const
	{
		guardSlow(UClass::GetSuperClass);
		return (UClass *)SuperField;
		unguardSlow;
	}
	UObject* GetDefaultObject()
	{
		guardSlow(UClass::GetDefaultObject);
		check(Defaults.Num()==GetPropertiesSize());
		return (UObject*)&Defaults(0);
		unguardobjSlow;
	}
	const UObject* GetDefaultObject() const
	{
		guardSlow(UClass::GetDefaultObjectConst);
		check(Defaults.Num()==GetPropertiesSize());
		return (UObject*)&Defaults(0);
		unguardobjSlow;
	}
	class AActor* GetDefaultActor()
	{
		guardSlow(UClass::GetDefaultActor);
		check(Defaults.Num());
		return (AActor*)&Defaults(0);
		unguardobjSlow;
	}

private:
	// Hide IsA because calling IsA on a class almost always indicates
	// an error where the caller should use IsChildOf.
	UBOOL IsA( UClass* Parent ) const {return UObject::IsA(Parent);}
};

/*-----------------------------------------------------------------------------
	UConst.
-----------------------------------------------------------------------------*/

//
// An UnrealScript constant.
//
class CORE_API UConst : public UField
{
	DECLARE_CLASS(UConst,UField,0,Core)
	DECLARE_WITHIN(UStruct)
	NO_DEFAULT_CONSTRUCTOR(UConst)

	// Variables.
	FString Value;

	// Constructors.
	UConst( UConst* InSuperConst, const TCHAR* InValue );

	// UObject interface.
	void Serialize( FArchive& Ar );

	// UConst interface.
	UConst* GetSuperConst() const
	{
		guardSlow(UConst::GetSuperStruct);
		checkSlow(!SuperField||SuperField->IsA(UConst::StaticClass()));
		return (UConst*)SuperField;
		unguardSlow;
	}
};

/*-----------------------------------------------------------------------------
	TFieldIterator.
-----------------------------------------------------------------------------*/

//
// For iterating through a linked list of fields.
//
template <class T> class TFieldIterator {
public:
	TFieldIterator(UStruct* InStruct)
		: Struct(InStruct)
		, Field(InStruct ? InStruct->Children : NULL) {
		IterateToNext();
	}
	operator UBOOL() {
		return Field != NULL;
	}
	void operator++() {
		checkSlow(Field);
		Field = Field->Next;
		IterateToNext();
	}
	T* operator*() {
		checkSlow(Field);
		return (T*)Field;
	}
	T* operator->() {
		checkSlow(Field);
		return (T*)Field;
	}
	UStruct* GetStruct() {
		return Struct;
	}
protected:
	void IterateToNext() {
		while (Struct) {
			while (Field) {
				if (Field->IsA(T::StaticClass()))
					return;
				Field = Field->Next;
			}
			Struct = Struct->GetInheritanceSuper();
			if (Struct)
				Field = Struct->Children;
		}
	}
	UStruct* Struct;
	UField* Field;
};

template <class T, EClassFlags Flag> class TFieldFlagIterator // sjs
{
public:
	TFieldFlagIterator(UStruct* InStruct)
		: Struct(InStruct)
		, Field(InStruct ? InStruct->Children : NULL) {
		IterateToNext();
	}
	inline operator UBOOL() {
		return Field != NULL;
	}
	inline void operator++() {
		checkSlow(Field);
		Field = Field->Next;
		IterateToNext();
	}
	inline T* operator*() {
		checkSlow(Field);
		return (T*)Field;
	}
	inline T* operator->() {
		checkSlow(Field);
		return (T*)Field;
	}
	inline UStruct* GetStruct() {
		return Struct;
	}
protected:
	inline void IterateToNext() {
		while (Struct) {
			while (Field) {
				if ((Field->GetClass()->ClassFlags & Flag))
					return;
				Field = Field->Next;
			}
			Struct = Struct->GetInheritanceSuper();
			if (Struct)
				Field = Struct->Children;
		}
	}
	UStruct* Struct;
	UField* Field;
};

#include "UnForcePacking_end.h"

/*----------------------------------------------------------------------------
	Core templates.
----------------------------------------------------------------------------*/

// Hash function.
inline DWORD GetTypeHash(const UObject* A) {
	return A ? A->GetIndex() : 0;
}

// Parse an object name in the input stream.
template< class T > UBOOL ParseObject(const TCHAR* Stream, const TCHAR* Match, T*& Obj, UObject* Outer) {
	return ParseObject(Stream, Match, T::StaticClass(), *(UObject**)&Obj, Outer);
}

// Find an optional object.
template< class T > T* FindObject(UObject* Outer, const TCHAR* Name, UBOOL ExactClass = 0) {
	return (T*)UObject::StaticFindObject(T::StaticClass(), Outer, Name, ExactClass);
}

// Find an object, no failure allowed.
template< class T > T* FindObjectChecked(UObject* Outer, const TCHAR* Name, UBOOL ExactClass = 0) {
	return (T*)UObject::StaticFindObjectChecked(T::StaticClass(), Outer, Name, ExactClass);
}

// Dynamically cast an object type-safely.
class UClass;
template< class T > T* Cast(UObject* Src) {
	return Src && Src->IsA(T::StaticClass()) ? (T*)Src : NULL;
}
template< class T, class U > T* CastChecked(U* Src) {
	if (!Src || !Src->IsA(T::StaticClass()))
		appErrorf(TEXT("Cast of %s to %s failed"), Src ? Src->GetFullName() : TEXT("NULL"), T::StaticClass()->GetName());
	return (T*)Src;
}
template< class T, EClassFlags Flag > T* FlagCast(UObject* Src) // sjs
{
	return Src && (Src->GetClass()->ClassFlags & Flag) ? (T*)Src : NULL;
}

// Construct an object of a particular class.
template< class T > T* ConstructObject(UClass* Class, UObject* Outer = (UObject*)-1, FName Name = NAME_None, DWORD SetFlags = 0) {
	check(Class->IsChildOf(T::StaticClass()));
	if (Outer == (UObject*)-1)
		Outer = UObject::GetTransientPackage();
	return (T*)UObject::StaticConstructObject(Class, Outer, Name, SetFlags);
}

// Load an object.
template< class T > T* LoadObject(UObject* Outer, const TCHAR* Name, const TCHAR* Filename, DWORD LoadFlags, UPackageMap* Sandbox) {
	return (T*)UObject::StaticLoadObject(T::StaticClass(), Outer, Name, Filename, LoadFlags, Sandbox);
}

// Load a class object.
template< class T > UClass* LoadClass(UObject* Outer, const TCHAR* Name, const TCHAR* Filename, DWORD LoadFlags, UPackageMap* Sandbox) {
	return UObject::StaticLoadClass(T::StaticClass(), Outer, Name, Filename, LoadFlags, Sandbox);
}

// Get default object of a class.
template< class T > T* GetDefault() {
	return (T*)&T::StaticClass()->Defaults(0);
}

/*-----------------------------------------------------------------------------
	The End.
-----------------------------------------------------------------------------*/

